#include <stdio.h>
#include <string.h>
#include "esp_system.h"
#include "kidbright32.h"
#include "line_follower.h"

// GPA 0..7 = DETECTOR 0..7
// GPA    7 6 5 4 3 2 1 0
//        | | | | | | | |____ detector bit 0 (left most from car front)
//        | | | | | | |______ detector bit 1
//        | | | | | |________ detector bit 2
//        | | | | |__________ detector bit 3
//        | | | |____________ detector bit 4
//        | | |______________ detector bit 5
//        | |________________ detector bit 6
//        |__________________ detector bit 7

// GPB    7 6 5 4 3 2 1 0
//        | | | | | | | |____ detector bit 8
//        | | | | | | |______ detector bit 9
//        | | | | | |________ detector bit 10 (right most from car front)
//        | | | | |__________ detector aux bit
//        | | | |____________ output aux 1
//        | | |______________ output aux 2
//        | |________________ led enable (active high)
//        |__________________ motor feedback enable (active high)

//
//  bit no.   0  1  2  3  4    5    6  7  8  9  10
//  weight   -6 -5 -4 -3 -2    1    2  3  4  5  6

// mcp23017 registers
#define MCP23017_REG_IODIRA				0x00
#define MCP23017_REG_IODIRB				0x01
#define MCP23017_REG_GPIOA				0x12
#define MCP23017_REG_GPIOB				0x13

LINE_FOLLOWER::LINE_FOLLOWER(int bus_ch, int dev_addr) {
	channel = bus_ch;
	address = dev_addr;
}

void LINE_FOLLOWER::init(void) {
	gpa_value = 0;
	gpb_value = 0;
	state = s_detect;
}

int LINE_FOLLOWER::prop_count(void) {
	// not supported
	return 0;
}

bool LINE_FOLLOWER::prop_name(int index, char *name) {
	// not supported
	return false;
}

bool LINE_FOLLOWER::prop_unit(int index, char *unit) {
	// not supported
	return false;
}

bool LINE_FOLLOWER::prop_attr(int index, char *attr) {
	// not supported
	return false;
}

bool LINE_FOLLOWER::prop_read(int index, char *value) {
	// not supported
	return false;
}

bool LINE_FOLLOWER::prop_write(int index, char *value) {
	// not supported
	return false;
}

void LINE_FOLLOWER::process(Driver *drv) {
	I2CDev *i2c = (I2CDev *)drv;
	uint8_t data[3];

	switch (state) {
		case s_detect:
			// detect i2c device
			if (i2c->detect(channel, address) == ESP_OK) {
				// init mcp23017
				data[0] = MCP23017_REG_GPIOA;
				data[1] = 0xff; // PA = 1111 1111 (input)
				//data[2] = 0x00; // PB = 0000 0000 (output)
				data[2] = 0xff; // PB = 1111 1111 (output/input)
				if (i2c->write(channel, address, data, 3) == ESP_OK) {
					data[0] = MCP23017_REG_IODIRA;
					data[1] = 0xff; // IODIRA = 1111 1111 (input)
					//data[2] = 0x00; // IODIRB = 0000 0000 (output)
					data[2] = 0x0f; // IODIRB = 0000 1111 (output/input)
					if (i2c->write(channel, address, data, 3) == ESP_OK) {
						// clear error flag
						error = false;
						state = s_read;

						// testbug
						tickcnt = get_tickcnt();
					}
					else {
						state = s_error;
					}
				}
				else {
					state = s_error;
				}
			}
			else {
				state = s_error;
			}
			break;

		case s_read:
			data[0] = MCP23017_REG_GPIOA;
			/*if (i2c->read(channel, address, &data[0], 1, &data[1], 1) == ESP_OK) {
				gpa_value = data[1];
				// set initialized flag
				initialized = true;
			}*/
			if (i2c->read(channel, address, &data[0], 1, &data[1], 2) == ESP_OK) {
				gpa_value = data[1];
				gpb_value = data[2] & 0x0f;
				// set initialized flag
				initialized = true;
			}
			else {
				state = s_error;
			}
			break;

		case s_error:
			// set error flag
			error = true;
			// clear initialized flag
			initialized = false;
			// get current tickcnt
			tickcnt = get_tickcnt();
			// goto wait and retry with detect state
			state = s_wait;
			break;

		case s_wait:
			// delay 1000ms before retry detect
			if (is_tickcnt_elapsed(tickcnt, 1000)) {
				state = s_detect;
			}
			break;
	}
}

int LINE_FOLLOWER::read(uint8_t bit) {
	// black = 1, white = 0
	if (bit <= 7) {
		return (((gpa_value ^ 0xff) >> bit) & 0x01);
	}
	else
	if ((bit >= 8) && (bit <= 15)) {
		return (((gpb_value ^ 0xff) >> (bit - 8)) & 0x01);
	}

	// default value
	return 0;
}

int LINE_FOLLOWER::read_value(void) {
	//return ((gpa_value ^ 0xff) & 0x1f); // 5-bit
	return (((gpb_value ^ 0xff) & 0x07) << 8) | ((gpa_value ^ 0xff) & 0xff); // 11-bit
}
