#include <stdio.h>
#include "app.h"
#include "util.h"
#include "debug.h"
#include "drv8833.h"
#include "ros.h"
#include "std_msgs/String.h" // publish string
#include "std_msgs/Float64.h" // subscribe float
#include "std_msgs/UInt32.h" // subscribe uint32
#include "geometry_msgs/Twist.h"
#include "lino_msgs/Velocities.h"

/*
roscore
rosrun rosserial_python serial_node.py _port:=/dev/ttyS4 _baud:=57600

[list node]
rosnode list

[list topic]
rostopic list

[subscribe]
rostopic echo /chatter

[publish]
rostopic pub /rpm_a std_msgs/Float64 50.5
rostopic pub /mc_timeout_ms std_msgs/UInt32 10
*/

//#define RPM_SUBSCRIBE_ENABLED
//#define CHATTER_TEST
//#define DIAG_TEST
#ifdef DIAG_TEST
#define DIAG_SPEED		30
//#define DIAG_SPEED		120
#endif

extern IWDG_HandleTypeDef hiwdg;
extern UART_HandleTypeDef hlpuart1; // debug uart

#ifdef RPM_SUBSCRIBE_ENABLED
void rpm_a_callback(const std_msgs::Float64& msg) {
	// set motor_r rpm
	drv8833_rpm_r(msg.data);
}

void rpm_b_callback(const std_msgs::Float64& msg) {
	// set motor_l rpm
	drv8833_rpm_l(msg.data);
}
#endif

void cmd_vel_callback(const geometry_msgs::Twist& cmd_msg) {
	// move base
	drv8833_base_move(cmd_msg.linear.x, cmd_msg.linear.y, cmd_msg.angular.z);
}

void mc_timeout_ms_callback(const std_msgs::UInt32& msg) {
	// set motor control timeout
	drv8833_mc_timeout_ms(msg.data);
}

ros::NodeHandle nh;
static uint32_t ros_tickcnt;

// subscribe
#ifdef RPM_SUBSCRIBE_ENABLED
ros::Subscriber<std_msgs::Float64> rpm_a_sub("rpm_a", &rpm_a_callback);
ros::Subscriber<std_msgs::Float64> rpm_b_sub("rpm_b", &rpm_b_callback);
#endif
ros::Subscriber<geometry_msgs::Twist> cmd_vel_sub("cmd_vel", &cmd_vel_callback);
ros::Subscriber<std_msgs::UInt32> mc_timeout_ms_sub("mc_timeout_ms", &mc_timeout_ms_callback);

// Publisher
lino_msgs::Velocities raw_vel_msg;
ros::Publisher raw_vel_pub("raw_vel", &raw_vel_msg);
static uint32_t raw_vel_tickcnt;

#ifdef CHATTER_TEST
std_msgs::String str_msg;
ros::Publisher chatter("chatter", &str_msg);
static uint32_t chatter_tickcnt;
static uint32_t msg_id = 0;
char hello[] = "Hello world!";
#endif

#ifdef DIAG_TEST
typedef enum {
	s_diag_start,
	s_diag_motor_test,
	s_diag_idle
} diag_state_t;

static struct {
	diag_state_t state;
	uint8_t test_index;
	uint32_t tickcnt, delay;
} diag;
#endif

void HAL_UART_TxCpltCallback(UART_HandleTypeDef *huart) {
	//debug_putstrln("HAL_UART_TxCpltCallback");
	nh.getHardware()->flush();
}

void HAL_UART_RxCpltCallback(UART_HandleTypeDef *huart) {
	//debug_putstrln("HAL_UART_RxCpltCallback");
	nh.getHardware()->reset_rbuf();
}

void app_init(void) {
	// get cpu id
	//CpuID[0]=*(uint32_t *)(UID_BASE);
	//CpuID[1]=*(uint32_t *)(UID_BASE + 4);
	//CpuID[2]=*(uint32_t *)(UID_BASE + 8);

	debug_init(&hlpuart1);

	drv8833_init();
	drv8833_rpm_r(0); // test values = 0, 30, 100, -30, -100
	drv8833_rpm_l(0);

	// init ros node
	nh.initNode();
	// publish
	raw_vel_msg.linear_x = 0;
	raw_vel_msg.linear_y = 0;
	raw_vel_msg.angular_z = 0;
	// subscribe
#ifdef RPM_SUBSCRIBE_ENABLED
	nh.subscribe(rpm_a_sub);
	nh.subscribe(rpm_b_sub);
#endif
	nh.subscribe(mc_timeout_ms_sub);
	nh.subscribe(cmd_vel_sub);

	// advertise
#ifdef CHATTER_TEST
	nh.advertise(chatter);
	chatter_tickcnt = HAL_GetTick();
#endif
	nh.advertise(raw_vel_pub);
	raw_vel_tickcnt = HAL_GetTick();

	// ros spinonce tick count
	ros_tickcnt = HAL_GetTick();

#ifdef DIAG_TEST
	// diagnostic process
	diag.state = s_diag_start;
#endif

	debug_putstr("kidbot-g4 v1.1\r\n");
}

#ifdef DIAG_TEST
void diag_proc(void) {
	switch (diag.state) {
		case s_diag_start:
			diag.test_index = 0;
			diag.delay = 5000;
			diag.tickcnt = HAL_GetTick() + diag.delay - 1000;
			diag.state = s_diag_motor_test;
			break;

		case s_diag_motor_test:
			if (is_tickcnt_elaspsed(diag.tickcnt, diag.delay)) {
				switch (diag.test_index) {
					case 0:
						drv8833_rpm_r(DIAG_SPEED); // test values = 0, 30, 100, -30, -100
						drv8833_rpm_l(0);
						diag.delay = 10000;
						break;

					case 1:
						drv8833_rpm_r(0);
						drv8833_rpm_l(0);
						diag.delay = 1000;
						break;

					case 2:
						drv8833_rpm_r(-1 * DIAG_SPEED);
						drv8833_rpm_l(0);
						diag.delay = 10000;
						break;

					case 3:
						drv8833_rpm_r(0);
						drv8833_rpm_l(0);
						diag.delay = 1000;
						break;

					case 4:
						drv8833_rpm_r(0);
						drv8833_rpm_l(DIAG_SPEED);
						diag.delay = 10000;
						break;

					case 5:
						drv8833_rpm_r(0);
						drv8833_rpm_l(0);
						diag.delay = 1000;
						break;

					case 6:
						drv8833_rpm_r(0);
						drv8833_rpm_l(-1 * DIAG_SPEED);
						diag.delay = 10000;
						break;

					default:
						drv8833_rpm_r(0);
						drv8833_rpm_l(0);
						diag.state = s_diag_idle;
						break;
				}

				diag.test_index++;
				diag.tickcnt = HAL_GetTick();
			}
			break;

		case s_diag_idle:
			break;
	}
}
#endif

void app_proc(void) {
	float linear_vel_x, linear_vel_y, angular_vel_z;

	// publish raw_vel_msg
	if ((HAL_GetTick() - raw_vel_tickcnt) >= 25) {
		raw_vel_tickcnt = HAL_GetTick();

		drv8833_base_get(&linear_vel_x, &linear_vel_y, &angular_vel_z);
		raw_vel_msg.linear_x = linear_vel_x;
		raw_vel_msg.linear_y = linear_vel_y;
		raw_vel_msg.angular_z = angular_vel_z;
		raw_vel_pub.publish(&raw_vel_msg);
	}

#ifdef CHATTER_TEST
	char msg[64];
	// test chatter publish every 1 second
	if (is_tickcnt_elaspsed(chatter_tickcnt, 1000)) {
		chatter_tickcnt = HAL_GetTick();
		// test uart dma tx
		//strcpy(msg, "Hello World!\r\n");
		//HAL_UART_Transmit_DMA(&huart1, (uint8_t *)msg, strlen(msg));

		msg_id++;
		snprintf(msg, sizeof(msg) - 1, "chatter %ld - %s", msg_id, hello);
		str_msg.data = msg;
		chatter.publish(&str_msg);
	}
#endif

	// ros spin once process
	if (is_tickcnt_elaspsed(ros_tickcnt, 100)) {
		ros_tickcnt = HAL_GetTick();
		GREEN_TOGGLE();
		nh.spinOnce();

#ifdef DIAG_TEST
		RED_TOGGLE();
#endif

		// test ros timer
		//debug_writedecln(nh.getHardware()->time());
	}

	// motor drive process
	drv8833_proc();
	// debug print process
	debug_proc();

#ifdef DIAG_TEST
	// diagnostic process
	diag_proc();
#endif

	// watchdog reset
	HAL_IWDG_Refresh(&hiwdg);
}
