#include <stdio.h>
#include "app.h"
#include "arm_math.h"
#include "util.h"
#include "debug.h"
#include "drv8833.h"
// https://github.com/linorobot/linorobot/tree/master/teensy/firmware/lib/kinematics
#include "Kinematics.h"

/*
[Project 03- STM32F4xx PID controller]
https://stm32f4-discovery.net/2014/11/project-03-stm32f4xx-pid-controller

[stm32 counter]
https://letanphuc.net/2015/06/stm32f0-timer-tutorial-and-counter-tutorial

[motor 6 v, 150:1 gear, 220 rpm, 2.0 kg.cm.]
https://www.pololu.com/product/3076/specs

[encoder 12 CPR]
https://www.pololu.com/product/3081/specs

[motor, TIM3 = 16-bit]
PWM_MR_P	= PB4
PWM_MR_N	= PA6
PWM_ML_P	= PB5
PWM_ML_N	= PA4

[control pins]
FAULT		= PA0
SLEEP		= PA1

[encoder, timer16]
COUNTER_R	= PA11
COUNTER_L	= PA8

### using rostopic to publish to negative rpm
rostopic pub /rpm_r std_msgs/Float64 50
rostopic pub /rpm_l std_msgs/Float64 -- -50

### twist message test
rostopic pub -r 10 /cmd_vel geometry_msgs/Twist  '{linear:  {x: 0.1, y: 0.0, z: 0.0}, angular: {x: 0.0,y: 0.0,z: 0.0}}'
rostopic pub -r 10 /cmd_vel geometry_msgs/Twist  '{linear:  {x: 0, y: 0.0, z: 0.0}, angular: {x: 0.0,y: 0.0,z: 0.0}}'

### keyboard control twist message test (twist test @30rpm)
sudo apt-get install ros-kinetic-teleop-twist-keyboard
roscore
sudo python /home/pi/src/reset_kibot_base.py
rosrun rosserial_python serial_node.py _port:=/dev/ttyS4 _baud:=57600
rosrun teleop_twist_keyboard teleop_twist_keyboard.py _speed:=0.07 _turn:=1.5
*/

// motor pwm
extern TIM_HandleTypeDef htim3;
TIM_HandleTypeDef *motor_pwm_tmr = &htim3;
static uint32_t motor_pwm_period;

// input capture encoder
extern TIM_HandleTypeDef htim16;
extern TIM_HandleTypeDef htim4;
extern TIM_HandleTypeDef htim1;

/*
// Interrupt line used for Stspin240 Fault interrupt
#define EXTI_FAULT_IRQN					(EXTI15_10_IRQn)
// GPIO Pin used for the Stspin240  Enable pin and Faults (over current detection and thermal shutdown)
#define EXTI_FAULT_PIN					(GPIO_PIN_10)
// GPIO port used for the Stspin240  Enable pin and Faults (over current detection and thermal shutdown)
#define EXTI_FAULT_PORT					(GPIOA)
// Flag interrupt priority
#define EXTI_FAULT_PRIORITY				(3)
*/

#define DRV8833_FAST_DECAY				GPIO_PIN_RESET // fast decay, the H-bridge is disabled and recirculation current flows through the body diodes

// pwm frequency
#define DRV8833_MOTOR_PWM_FREQ			(1000)
#define DRV8833_FORWARD_DIR				0
#define DRV8833_BACKWARD_DIR			1

// motor spec
#define MOTOR_GEAR_RATIO				150.58
#define MOTOR_PPR						3		// datasheet spec = 12, need to calibrate down to 3

// pid config
#define PID_LOOP_MS						50
// #define PID_PARAM_KP					0.3		// proportional
// #define PID_PARAM_KI					0.7		// integral
// #define PID_PARAM_KD					0.03	// derivative
#define PID_PARAM_KP					0.1		// proportional
#define PID_PARAM_KI					0.4		// integral
#define PID_PARAM_KD					0.03	// derivative

// both edge trig counter, 50ms calc window
// safety min rpm = 15 (counter = 5 to 7), teleop_twist_keyboard speed = 0.035, turn = 0.75
// safety max rpm = 120 (counter = 44 to 46), teleop_twist_keyboard speed = 0.28, turn = 6.0
//
// safety min rpm = 20, teleop_twist_keyboard speed = 0.046, turn = 0.96
// safety max rpm = 60, teleop_twist_keyboard speed = 0.137, turn = 2.87
#define COUNTER_EDGE_NUM				2		// rising/falling edge = 1, both edge = 2
//#define RPM_CALC_WINDOW_MS				20
#define RPM_CALC_WINDOW_MS				50

#define OOF_RPM_ERROR					50
#define OOF_RPM_TIME_MS					500
#define OOF_RPM_WAIT_MS					500

#define LINO_BASE						DIFFERENTIAL_DRIVE	// 2WD and Tracked robot w/ 2 motors
#define MAX_RPM							120					// motor's maximum RPM (safe speed derived from set 50% duty cycle when battery is full charged)
#define WHEEL_DIAMETER					0.043				// wheel's diameter in meters
#define LR_WHEELS_DISTANCE				0.095				// distance between left and right wheels 0.800
#define FR_WHEELS_DISTANCE				0.093				// distance between front and rear wheels. Ignore this if you're on 2WD/ACKERMANN
Kinematics kinematics(Kinematics::LINO_BASE, MAX_RPM, WHEEL_DIAMETER, FR_WHEELS_DISTANCE, LR_WHEELS_DISTANCE);

typedef struct {
	float duty;
	float set_point;
	float error;
	float speed;
	int direction, new_direction;
	arm_pid_instance_f32 pid;
	int out_of_control_ms;
	int out_of_control_wait_ms;
} motor_t;

static motor_t motors[DRV8833_MOTOR_NUM];
static motor_t *motor_r = &motors[DRV8833_MOTOR_R];
static motor_t *motor_l = &motors[DRV8833_MOTOR_L];
static uint32_t drv8833_tickcnt;
static uint32_t drv8833_mc_timeout, drv8833_mc_timeout_tickcnt;

void drv8833_gpio_output_config(GPIO_TypeDef *port, uint16_t pin, GPIO_PinState state) {
	GPIO_InitTypeDef GPIO_InitStruct;

	GPIO_InitStruct.Pin = pin;
	GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
	GPIO_InitStruct.Pull = GPIO_NOPULL;
	GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_MEDIUM;
	HAL_GPIO_WritePin(port, pin, state);
	HAL_GPIO_Init(port, &GPIO_InitStruct);
}

void drv8833_gpio_af_config(GPIO_TypeDef *port, uint16_t pin, uint8_t alternate) {
	GPIO_InitTypeDef GPIO_InitStruct;

    GPIO_InitStruct.Pin = pin;
    GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
    GPIO_InitStruct.Pull = GPIO_NOPULL;
    GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
    GPIO_InitStruct.Alternate = alternate;
    HAL_GPIO_Init(port, &GPIO_InitStruct);
}

void drv8833_pwm_r(float duty) {
	uint32_t period;

	// calc period
	if (duty > 100) {
		duty = 100;
	}
	period = ((duty * motor_pwm_period) / 100) + 1;
	__HAL_TIM_SET_COMPARE(motor_pwm_tmr, TIM_CHANNEL_1, period);
	motor_r->duty = duty;
}

void drv8833_pwm_l(float duty) {
	uint32_t period;

	// calc period
	if (duty > 100) {
		duty = 100;
	}
	period = ((duty * motor_pwm_period) / 100) + 1;
	__HAL_TIM_SET_COMPARE(motor_pwm_tmr, TIM_CHANNEL_2, period);
	motor_l->duty = duty;
}

void drv8833_duty_r(float duty) {
	if (duty == 0) {
		// set pwm period = 1
		__HAL_TIM_SET_COMPARE(motor_pwm_tmr, TIM_CHANNEL_1, 1);
		// MR_P = 1
		drv8833_gpio_output_config(MR_P_GPIO_Port, MR_P_Pin, DRV8833_FAST_DECAY);
		// MR_N = 1
		drv8833_gpio_output_config(MR_N_GPIO_Port, MR_N_Pin, DRV8833_FAST_DECAY);
	}
	else {
		if (motor_r->direction == DRV8833_FORWARD_DIR) {
			// MR_N = 0
			drv8833_gpio_output_config(MR_N_GPIO_Port, MR_N_Pin, DRV8833_FAST_DECAY);
			// setup pwm
			drv8833_pwm_r(duty);
			// MR_P = PWM
			drv8833_gpio_af_config(MR_P_GPIO_Port, MR_P_Pin, GPIO_AF2_TIM3);
		}
		else {
			// MR_P = 0
			drv8833_gpio_output_config(MR_P_GPIO_Port, MR_P_Pin, DRV8833_FAST_DECAY);
			// setup pwm
			drv8833_pwm_r(duty);
			// MR_N = PWM
			drv8833_gpio_af_config(MR_N_GPIO_Port, MR_N_Pin, GPIO_AF2_TIM3);
		}
	}
}

void drv8833_duty_l(float duty) {
	if (duty == 0) {
		// set pwm period = 1
		__HAL_TIM_SET_COMPARE(motor_pwm_tmr, TIM_CHANNEL_2, 1);
		// ML_P = 1
		drv8833_gpio_output_config(ML_P_GPIO_Port, ML_P_Pin, DRV8833_FAST_DECAY);
		// ML_N = 1
		drv8833_gpio_output_config(ML_N_GPIO_Port, ML_N_Pin, DRV8833_FAST_DECAY);
	}
	else {
		if (motor_l->direction == DRV8833_FORWARD_DIR) {
			// ML_N = 0
			drv8833_gpio_output_config(ML_N_GPIO_Port, ML_N_Pin, DRV8833_FAST_DECAY);
			// setup pwm
			drv8833_pwm_l(duty);
			// ML_P = PWM
			drv8833_gpio_af_config(ML_P_GPIO_Port, ML_P_Pin, GPIO_AF2_TIM3);
		}
		else {
			// ML_P = 0
			drv8833_gpio_output_config(ML_P_GPIO_Port, ML_P_Pin, DRV8833_FAST_DECAY);
			// setup pwm
			drv8833_pwm_l(duty);
			// ML_N = PWM
			drv8833_gpio_af_config(ML_N_GPIO_Port, ML_N_Pin, GPIO_AF2_TIM3);
		}
	}
}

void drv8833_start_r(void) {
	HAL_TIM_PWM_Start(motor_pwm_tmr, TIM_CHANNEL_1);
}

void drv8833_start_l(void) {
	HAL_TIM_PWM_Start(motor_pwm_tmr, TIM_CHANNEL_2);
}

void HAL_TIM_OC_DelayElapsedCallback(TIM_HandleTypeDef *htim) {
	int i;
	float error, duty;

	if (htim == &htim16) {
//		DEBUG0_TOGGLE;

		// stop counter MR without interrupt
		HAL_TIM_Base_Stop(&htim4);
		// stop counter ML without interrupt
		HAL_TIM_Base_Stop(&htim1);

		// calc speed MR
		uint16_t cnt_r = __HAL_TIM_GET_COUNTER(&htim4) / COUNTER_EDGE_NUM;
		if (cnt_r > 0) {
			motor_r->speed = (60 * cnt_r * 1000) / (MOTOR_GEAR_RATIO * MOTOR_PPR * RPM_CALC_WINDOW_MS);
		}
		else {
			motor_r->speed = 0;
		}

		// calc speed ML
		uint16_t cnt_l = __HAL_TIM_GET_COUNTER(&htim1) / COUNTER_EDGE_NUM;
		if (cnt_l > 0) {
			motor_l->speed = (60 * cnt_l * 1000) / (MOTOR_GEAR_RATIO * MOTOR_PPR * RPM_CALC_WINDOW_MS);
		}
		else {
			motor_l->speed = 0;
		}

		// clear counter MR
		__HAL_TIM_SET_COUNTER(&htim4, 0);
		// start counter MR without interrupt
		HAL_TIM_Base_Start(&htim4);
		// clear counter ML
		__HAL_TIM_SET_COUNTER(&htim1, 0);
		// start counter ML without interrupt
		HAL_TIM_Base_Start(&htim1);

		// run pid
		for (i = DRV8833_MOTOR_R; i < DRV8833_MOTOR_NUM; i++) {
			if (motors[i].set_point == 0) {
				error = 0;
				duty = 0;
				// set new duty cycle
				if (i == DRV8833_MOTOR_R) {
					drv8833_duty_r(duty);
				}
				else
				if (i == DRV8833_MOTOR_L) {
					drv8833_duty_l(duty);
				}
			}
			else {
				// error calc
				error = motors[i].set_point - motors[i].speed;
				// pid
				duty = arm_pid_f32(&motors[i].pid, error);
				// validate duty cycle range
				if (duty > 100) {
					duty = 100;
				}
				else
				if (duty < 0) {
					duty = 0;
				}

				// save current error
				motors[i].error = error;

				if (motors[i].out_of_control_wait_ms > 0) {
					motors[i].out_of_control_wait_ms -= RPM_CALC_WINDOW_MS;
				}
				else {
					// out of control check
					if (error > OOF_RPM_ERROR) {
						motors[i].out_of_control_ms += RPM_CALC_WINDOW_MS;
						// if over out of control
						if (motors[i].out_of_control_ms > OOF_RPM_TIME_MS) {
							duty = 0;
							motors[i].out_of_control_wait_ms = OOF_RPM_WAIT_MS;
						}
					}

					// set new duty cycle
					if (i == DRV8833_MOTOR_R) {
						drv8833_duty_r(duty);
					}
					else
					if (i == DRV8833_MOTOR_L) {
						drv8833_duty_l(duty);
					}
				}
			}
		}
	}
}

void drv8833_rpm_r(float rpm) {
	// disable timer16 interrupt
	HAL_NVIC_DisableIRQ(TIM1_UP_TIM16_IRQn);

	// check direction
	if (rpm >= 0) {
		motor_r->set_point = rpm;
		if (motor_r->direction != DRV8833_FORWARD_DIR) {
			// update new direction
			motor_r->direction = DRV8833_FORWARD_DIR;
			// stop motor MR
			drv8833_duty_r(0);
			// re-initialize pid
			arm_pid_init_f32(&motor_r->pid, 1);
		}
	}
	else {
		motor_r->set_point = (-1 * rpm);
		if (motor_r->direction != DRV8833_BACKWARD_DIR) {
			// update new direction
			motor_r->direction = DRV8833_BACKWARD_DIR;
			// stop motor MR
			drv8833_duty_r(0);
			// re-initialize pid
			arm_pid_init_f32(&motor_r->pid, 1);
		}
	}

	// enable timer16 interrupt
	HAL_NVIC_EnableIRQ(TIM1_UP_TIM16_IRQn);
}

void drv8833_rpm_l(float rpm) {
	// disable timer16 interrupt
	HAL_NVIC_DisableIRQ(TIM1_UP_TIM16_IRQn);

	// check direction
	if (rpm >= 0) {
		motor_l->set_point = rpm;
		if (motor_l->direction != DRV8833_FORWARD_DIR) {
			// update new direction
			motor_l->direction = DRV8833_FORWARD_DIR;
			// stop motor ML
			drv8833_duty_l(0);
			// re-initialize pid
			arm_pid_init_f32(&motor_l->pid, 1);
		}
	}
	else {
		motor_l->set_point = (-1 * rpm);
		if (motor_l->direction != DRV8833_BACKWARD_DIR) {
			// update new direction
			motor_l->direction = DRV8833_BACKWARD_DIR;
			// stop motor ML
			drv8833_duty_l(0);
			// re-initialize pid
			arm_pid_init_f32(&motor_l->pid, 1);
		}
	}

	// enable timer16 interrupt
	HAL_NVIC_EnableIRQ(TIM1_UP_TIM16_IRQn);
}

void drv8833_mc_timeout_ms(uint32_t timeout) {
	debug_putstr("set motor control timeout to ");
	debug_writedec(timeout);
	debug_putstrln(" ms");

	// reload motor control timeout tickcnt
	drv8833_mc_timeout_tickcnt = HAL_GetTick();
	// update motor control timeout
	drv8833_mc_timeout = timeout;
}

void drv8833_base_move(float linear_vel_x, float linear_vel_y, float angular_vel_z) {
	Kinematics::rpm req_rpm = kinematics.getRPM(linear_vel_x, linear_vel_y, angular_vel_z);
/*
	char buf[256];
	snprintf(buf, sizeof(buf) - 1, "linear xy = %f, %f, anular z = %f", linear_vel_x, linear_vel_y, angular_vel_z);
	debug_putstrln(buf);
	snprintf(buf, sizeof(buf) - 1, "req_rpm = %d, %d, %d, %d\r\n", req_rpm.motor1, req_rpm.motor2, req_rpm.motor3, req_rpm.motor4);
	debug_putstrln(buf);
*/

	// update setpoint
	drv8833_rpm_r((float)req_rpm.motor1);
	drv8833_rpm_l((float)req_rpm.motor2);

	// reload motor control timeout tickcnt
	drv8833_mc_timeout_tickcnt = HAL_GetTick();
}

void drv8833_base_get(float *linear_vel_x, float *linear_vel_y, float *angular_vel_z) {
	Kinematics::velocities current_vel;
/*	float rpm_r, rpm_l;

	rpm_r = 0;
	rpm_l = 0;

	// base forward
	if ((motor_r->direction == DRV8833_FORWARD_DIR) && (motor_l->direction == DRV8833_BACKWARD_DIR)) {
		rpm_r = motor_r->speed;
		rpm_l = motor_l->speed;
	}
	else
	// base backward
	if ((motor_r->direction == DRV8833_BACKWARD_DIR) && (motor_l->direction == DRV8833_FORWARD_DIR)) {
		rpm_r = -1 * motor_r->speed;
		rpm_l = -1 * motor_l->speed;
	}
	else
	// base counter clockwise
	if ((motor_r->direction == DRV8833_FORWARD_DIR) && (motor_l->direction == DRV8833_FORWARD_DIR)) {
		rpm_r = -1 * motor_r->speed;
		rpm_l = motor_l->speed;
	}
	else
	// base clockwise
	if ((motor_r->direction == DRV8833_BACKWARD_DIR) && (motor_l->direction == DRV8833_BACKWARD_DIR)) {
		rpm_r = motor_r->speed;
		rpm_l = -1 * motor_l->speed;
	}

	current_vel = kinematics.getVelocities(rpm_r, rpm_l, 0, 0);*/

	// get current rpm
	current_vel = kinematics.getVelocities(motor_r->speed, motor_l->speed, 0, 0);

	// return linear x/y, angular z
	*linear_vel_x = current_vel.linear_x;
	*linear_vel_y = current_vel.linear_y;
	*angular_vel_z = current_vel.angular_z;
}

void drv8833_init(void) {
	int i;

	// GPIO Ports Clock Enable
	__HAL_RCC_GPIOA_CLK_ENABLE();
	__HAL_RCC_GPIOB_CLK_ENABLE();
	__HAL_RCC_GPIOC_CLK_ENABLE();

/*	// Configure Stspin240 Enable pin
	// When this pin is set low, it is configured just before as
	// GPIO_MODE_OUTPUT_PP with GPIO_NOPULL
	// When this pin is set high, it is just after configured for FAULT
	// as GPIO_MODE_IT_FALLING with GPIO_PULLUP
	GPIO_InitStruct.Pin = DRV8833_FAULT_PIN;
	GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
	GPIO_InitStruct.Pull = GPIO_NOPULL;
	GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_MEDIUM;
	HAL_GPIO_Init(DRV8833_FAULT_PORT, &GPIO_InitStruct);
	HAL_GPIO_WritePin(DRV8833_FAULT_PORT, DRV8833_FAULT_PIN, GPIO_PIN_RESET);
	// Set Priority of External Line Interrupt used for the OCD OVT interrupt
	HAL_NVIC_SetPriority(EXTI_FAULT_IRQN, EXTI_FAULT_PRIORITY, 0);
	// Enable the External Line Interrupt used for the OCD OVT interrupt
	HAL_NVIC_EnableIRQ(EXTI_FAULT_IRQN);*/

	// config MOTOR_R hi-z output => MR_P/MR_N = 0
	drv8833_gpio_output_config(MR_P_GPIO_Port, MR_P_Pin, GPIO_PIN_RESET);
	drv8833_gpio_output_config(MR_N_GPIO_Port, MR_N_Pin, GPIO_PIN_RESET);
	// config MOTOR_B hi-z output => ML_P/ML_N = 0
	drv8833_gpio_output_config(ML_P_GPIO_Port, ML_P_Pin, GPIO_PIN_RESET);
	drv8833_gpio_output_config(ML_N_GPIO_Port, ML_N_Pin, GPIO_PIN_RESET);

	// init motor control parameters
	for (i = 0; i < DRV8833_MOTOR_NUM; i++) {
		// pid parameters
		motors[i].pid.Kp = PID_PARAM_KP;
		motors[i].pid.Ki = PID_PARAM_KI;
		motors[i].pid.Kd = PID_PARAM_KD;
		motors[i].set_point = 0;
		motors[i].error = 0;
		motors[i].speed = 0;
		motors[i].direction = DRV8833_FORWARD_DIR;
		motors[i].duty = 0;
		// init pid system, float32_t format
		arm_pid_init_f32(&motors[i].pid, 1);
	}

	// default no motor control timeout
	drv8833_mc_timeout = 0;

	// calc pwm period
	motor_pwm_period = ((HAL_RCC_GetPCLK1Freq() / (motor_pwm_tmr->Instance->PSC + 1)) / DRV8833_MOTOR_PWM_FREQ) - 1; // timer 3 clock = PCLK1
	// init motor pwm frequency
	__HAL_TIM_SET_AUTORELOAD(motor_pwm_tmr, motor_pwm_period);
	// int motor pwm duty cycle
	drv8833_duty_r(0);
	drv8833_start_r();
	drv8833_duty_l(0);
	drv8833_start_l();

	// de-assert sleep
	HAL_GPIO_WritePin(MOTOR_SLEEP_GPIO_Port, MOTOR_SLEEP_Pin, GPIO_PIN_SET);
	HAL_Delay(1);

	// set compare no output interval
	htim16.Instance->ARR = (RPM_CALC_WINDOW_MS * 1000) - 1;
	// start output compare no output
	HAL_TIM_OC_Start_IT(&htim16, TIM_CHANNEL_1);

	// clear counter MR
	__HAL_TIM_SET_COUNTER(&htim4, 0);
	// start counter MR without interrupt
	HAL_TIM_Base_Start(&htim4);

	// clear counter ML
	__HAL_TIM_SET_COUNTER(&htim1, 0);
	// start counter ML without interrupt
	HAL_TIM_Base_Start(&htim1);

	motor_r->set_point = 0; // rpm
	motor_r->direction = DRV8833_FORWARD_DIR;
	motor_r->out_of_control_ms = 0;
	motor_l->set_point = 0; // rpm
	motor_l->direction = DRV8833_FORWARD_DIR;
	motor_l->out_of_control_ms = 0;

	drv8833_tickcnt = HAL_GetTick();
}

void drv8833_proc() {
	char buf[64];

	// print motor speed
	if (is_tickcnt_elaspsed(drv8833_tickcnt, 500)) {
		drv8833_tickcnt = HAL_GetTick();
		// print control result if set point is set
		if (motors[DRV8833_MOTOR_R].set_point) {
			snprintf(buf, sizeof(buf) - 1, "MR = %2.2f, %2.2f, %2.2f, %2.2f", motors[DRV8833_MOTOR_R].set_point, motors[DRV8833_MOTOR_R].speed, motors[DRV8833_MOTOR_R].error, motors[DRV8833_MOTOR_R].duty);
			debug_putstrln(buf);
		}
		if (motors[DRV8833_MOTOR_L].set_point) {
			snprintf(buf, sizeof(buf) - 1, "ML = %2.2f, %2.2f, %2.2f, %2.2f", motors[DRV8833_MOTOR_L].set_point, motors[DRV8833_MOTOR_L].speed, motors[DRV8833_MOTOR_L].error, motors[DRV8833_MOTOR_L].duty);
			debug_putstrln(buf);
		}
	}

	// check motor control timeout
	if (drv8833_mc_timeout > 0) {
		// if one or both motors has been driving
		if ((motor_r->set_point != 0) || (motor_l->set_point != 0)) {
			// is tick count elasped
			if (is_tickcnt_elaspsed(drv8833_mc_timeout_tickcnt, drv8833_mc_timeout)) {
				debug_putstrln("motor control is timeout, stop all motors");
				// stop motor control
				drv8833_rpm_r(0);
				drv8833_rpm_l(0);
			}
		}
	}
}
